/*
	AppHack - An Application Bundle Editor and Application Theme Tool for Mac OS X

    Copyright (C) 2004-2006 	Sveinbjorn Thordarson <sveinbjornt@simnet.is>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#import "AppHackController.h"
#import "STUtil.h"
#import "NSString+CarbonFSSpecCreation.h"

@implementation AppHackController

- (id)init
{
	if (self = [super init]) 
	{

    }
    return self;
}

- (void)dealloc
{
	[docTypeList release];
	[toolbar release];
    [super dealloc];
}

+ (void)initialize 
{ 
	NSDictionary *registrationDefaults;
    registrationDefaults = [NSDictionary dictionaryWithContentsOfFile:
        [[NSBundle mainBundle] pathForResource: @"RegistrationDefaults" ofType: @"plist"]];
    [[NSUserDefaults standardUserDefaults] registerDefaults: registrationDefaults];
}

- (void)awakeFromNib
{
	//add the toolbar
	toolbar = [[NSToolbar alloc] initWithIdentifier: @"com.sveinbjornt.apphack"];
	[toolbar setDelegate: self];
	[toolbar setAllowsUserCustomization: YES];
	[toolbar setAutosavesConfiguration: YES];
	[toolbar setVisible: NO];
	[window setToolbar: toolbar];
	
	//add menu item actions for toolbar
	[showToolbarMenuItem setTarget: toolbar];
	[showToolbarMenuItem setAction: @selector(setVisible:)];
	[showToolbarMenuItem setEnabled: NO];
	[customizeToolbarMenuItem setTarget: toolbar];
	[customizeToolbarMenuItem setAction: @selector(runCustomizationPalette:)];
	[showToolbarMenuItem setEnabled: NO];
	
	//preferences
	[self setIconSize: [iconSizePopupMenu indexOfSelectedItem]+1];
	
	
	
}

/*****************************************
 - Handler for when app is done launching
 - Set up the window and stuff like that
*****************************************/
- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
	if ([[NSUserDefaults standardUserDefaults] boolForKey: @"promptOnLaunch"])
		[self open: self];
}

/*****************************************
 - Handler for dragged files and/or files opened via the Finder
*****************************************/

- (BOOL)application:(NSApplication *)theApplication openFile:(NSString *)filename
{
	BOOL	isDir = NO;

	if([[NSFileManager defaultManager] fileExistsAtPath: filename isDirectory: &isDir])
	{
		[self loadApp: filename];
		return(YES);
	}
	else
		return(NO);
}

#pragma mark -

- (IBAction)open:(id)sender
{
	NSArray	*fileTypes = [NSArray arrayWithObject: @"app"];
	
	//set icon size from prefs
	[self setIconSize: [iconSizePopupMenu indexOfSelectedItem]+1];
	
	//create open panel
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	//run open panel
    if ([oPanel runModalForTypes: fileTypes] == NSOKButton) 
			[self loadApp: [oPanel filename]];
}

#pragma mark -

- (void)setIconSize: (int)size
{
	[docTypesDataBrowser setAllowsColumnResizing: YES];
	[docTypesDataBrowser setRowHeight: size*16];
	[docTypesDataBrowser setAllowsColumnResizing: NO];
	
	
	NSArray *columns = [docTypesDataBrowser tableColumns];
	[[columns objectAtIndex: 0] setWidth: (size*16)];
	
	if ((int)[docTypesDataBrowser rowHeight] >= kMaxIconSize)
		[increaseIconSizeButton setEnabled: NO];
	else
		[increaseIconSizeButton setEnabled: YES];
	
	[decreaseIconSizeButton setEnabled: YES];
}

- (IBAction)increaseIconSize:(id)sender
{
	[docTypesDataBrowser setAllowsColumnResizing: YES];
	[docTypesDataBrowser setRowHeight: [docTypesDataBrowser rowHeight]+16];
	[docTypesDataBrowser setAllowsColumnResizing: NO];
	
	NSArray *columns = [docTypesDataBrowser tableColumns];
	[[columns objectAtIndex: 0] setWidth: [[columns objectAtIndex: 0] width]+16];
	
	if ((int)[docTypesDataBrowser rowHeight] >= kMaxIconSize)
		[increaseIconSizeButton setEnabled: NO];
	else
		[increaseIconSizeButton setEnabled: YES];
	
	[decreaseIconSizeButton setEnabled: YES];
}

- (IBAction)decreaseIconSize:(id)sender
{	
	[docTypesDataBrowser setRowHeight: [docTypesDataBrowser rowHeight]-16];
	
	[docTypesDataBrowser setAllowsColumnResizing: YES];
	NSArray *columns = [docTypesDataBrowser tableColumns];
	[[columns objectAtIndex: 0] setWidth: [[columns objectAtIndex: 0] width]-16];
	[docTypesDataBrowser setAllowsColumnResizing: NO];

		[increaseIconSizeButton setEnabled: YES];
	
	if ((int)[docTypesDataBrowser rowHeight] <= 16)
		[decreaseIconSizeButton setEnabled: NO];
	

}

#pragma mark -


- (IBAction)getFileInfo:(id)sender
{
	NSTask *task = [[NSTask alloc] init];
	NSString	*script = [NSString stringWithFormat: @"tell application \"Finder\" to open information window of item (POSIX file \"%@\")", [appBundle bundlePath]];
	NSMutableArray *args = [NSMutableArray array];

	//launch finder
	[[NSWorkspace sharedWorkspace] launchApplication: @"Finder"];
	
	//create arguments array
    [args addObject: @"-e"];
	[args addObject: script];
	
	//set up task and launch it
	[task setLaunchPath:@"/usr/bin/osascript"];
	[task setArguments:args];
    [task launch];
	[task release];
}


- (IBAction)showPackageContents:(id)sender
{
	BOOL		isDir;
	
	NSString	*path = [[appBundle bundlePath] stringByAppendingString:@"/Contents/"];
	if ([[NSFileManager defaultManager] fileExistsAtPath: path isDirectory:&isDir] && isDir) 
		[[NSWorkspace sharedWorkspace] selectFile: nil inFileViewerRootedAtPath:path];
}

- (IBAction)revealApplication:(id)sender
{
	BOOL		isDir;

	if ([[NSFileManager defaultManager] fileExistsAtPath: [appBundle bundlePath] isDirectory:&isDir]) 
		[[NSWorkspace sharedWorkspace] selectFile: [appBundle bundlePath] inFileViewerRootedAtPath: nil];
}

#pragma mark -

- (IBAction)importBundlePlist:(id)sender
{

}

- (IBAction)exportBundlePlist:(id)sender
{
	//make sure property list exists
	if(![[NSFileManager defaultManager] fileExistsAtPath: infoPlistPath isDirectory: nil])
		[STUtil alert: @"Incomplete Application Bundle" subText: [NSString stringWithFormat: @"No Info.plist file could be found within the application bundle", [appBundle bundlePath]]];
	else
	{
		//prompt for placement and copy it over there
		NSSavePanel *sPanel = [NSSavePanel savePanel];
		[sPanel setPrompt:@"Save"];
		[sPanel setTitle:@"Save Bundle Property List"];
		if ([sPanel runModalForDirectory:nil file: [infoPlistPath lastPathComponent] ] == NSOKButton) 
		{
				NSString *destPath = [sPanel filename];
				[[NSFileManager defaultManager] copyPath: infoPlistPath toPath: destPath handler: nil];
		}
	}
}

#pragma mark -

- (IBAction)importIconSet:(id)sender
{
	short i, j, changed;
	//file types acceptable as custom icon data
	NSArray *fileTypes = [NSArray arrayWithObjects: @"iconset", nil];
	NSMutableArray	*iconPaths = [[[NSMutableArray alloc] initWithCapacity: 255] autorelease];
	
	// Prompt for icon set selection
	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setTitle: @"Import Icon Set"];
	[oPanel setPrompt:@"Import"];
    [oPanel setAllowsMultipleSelection:NO];
		
	//run open panel
    if ([oPanel runModalForDirectory:nil file:nil types:fileTypes] != NSOKButton) 
		return;
	
	// Get all the files in the iconset directory of type .icns
	NSString *file;
	NSDirectoryEnumerator *enumerator = [[NSFileManager defaultManager] enumeratorAtPath: [oPanel filename]];
	while (file = [enumerator nextObject]) 
	{
		if ([[file pathExtension] isEqualToString:@"icns"])
			[iconPaths addObject: [NSString stringWithFormat: @"%@/%@", [oPanel filename], file]];
	}
	
	//  If there are no such files, we return warning
	if ([iconPaths count] <= 0)
	{
		[STUtil alert:@"Empty Icon Set" subText: @"The Icon Set you selected contains no icons."];
		return;
	}
		
	// for every icon in the imported set
	for (i = 0; i < [iconPaths count]; i++)
	{
		NSString *iconPath = [iconPaths objectAtIndex: i];
		NSString *importedIconName = [[STUtil cutSuffix: [iconPath lastPathComponent]] retain];
		
		// see if it matches the application icon
		if ([[STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] isEqualToString: importedIconName])
		{
			NSImage	*importedAppIcon = [[[NSImage alloc] initWithContentsOfFile: iconPath] autorelease];
			if (importedAppIcon != NULL)
				[appIconView setImage: importedAppIcon];
		}
		
		// for every icon named in our property list -- see if it matches a document icon
		for (j = 0; j < [[docTypeList array] count]; j++)
		{
			NSString *iconName = [STUtil cutSuffix: [[[[docTypeList array] objectAtIndex: j ] dict] objectForKey: @"CFBundleTypeIconFile"]];
			if (iconName == NULL)
				continue;
		
			//NSLog(@"Comparing %@ to %@", iconName, importedIconName);
			// if our property list icon has the same name as our imported icon
			if ([iconName isEqualToString: importedIconName])
			{
				
				//we set that icon to the imported icon
				NSImage	*importedIcon = [[[NSImage alloc] initWithContentsOfFile: iconPath] autorelease];
				if (importedIcon != NULL)
				{
						[[[docTypeList array] objectAtIndex: j ] setIcon: importedIcon];
						changed = YES;
						//NSLog(@"Set new icon for %@", iconName);
				}
			}
		}
	}
	
	// if something changed, we need to update the doc types data browser
	if (changed)
		[docTypesDataBrowser reloadData];

}

- (IBAction)exportAllIcons:(id)sender
{
	short				i;
	NSMutableArray		*iconPaths = [[[NSMutableArray alloc] initWithCapacity: 255] autorelease];
	
	//application icon
	NSString	*appIconPath = [appBundle pathForResource: [STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] ofType: @"icns"];
	if (appIconPath != NULL)
		[iconPaths addObject: appIconPath];
	
	//find the document icons
	for (i = 0; i < [[docTypeList array] count]; i++)
	{
		NSString	*iconFileName = [[[[docTypeList array] objectAtIndex: i] dict] objectForKey: @"CFBundleTypeIconFile"];
		if (iconFileName != NULL)
		{
			NSString	*thePath = [appBundle pathForResource: [STUtil cutSuffix: iconFileName]  ofType: @"icns"];
			if (thePath != NULL)
				[iconPaths addObject: thePath];
		}
	}
	
	// If there are some icon files
	if ([iconPaths count] <= 0)
	{
		[STUtil alert: @"No icon files found" subText: @"AppHack was unable to locate any icon files for any document types within the application bundle."];
		return;
	}
	
	//prompt for location to create iconset folder
	NSSavePanel *sPanel = [NSSavePanel savePanel];
	[sPanel setPrompt:@"Save"];
	[sPanel setTitle:@"Save Application Icons"];
	if ([sPanel runModalForDirectory:nil file: [NSString stringWithFormat: @"%@.iconset", [STUtil cutSuffix: [window title]] ] ] == NSOKButton) 
	{
		NSString *folderPath = [sPanel filename];
		[[NSFileManager defaultManager] createDirectoryAtPath: folderPath attributes:nil];
		for (i = 0; i < [iconPaths count]; i++)
		{
			NSString	*destPath = [NSString stringWithFormat: @"%@/%@", folderPath, [[iconPaths objectAtIndex: i] lastPathComponent]];
			[[NSFileManager defaultManager] copyPath: [iconPaths objectAtIndex: i] toPath: destPath handler: nil];
		}
		//set custom icon for iconset folder
		IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [NSImage imageNamed: @"IconsetIcon"]] autorelease];
		[iconFam setAsCustomIconForDirectory: folderPath];
	}
}



- (IBAction)appRoleChanged:(id)sender
{
	int	selectedDocType = [docTypesDataBrowser selectedRow];
	

	if ([[appRoleRadioButtons selectedCell] tag] == 0 || [[[[docTypeList array] objectAtIndex: selectedDocType ] dict] objectForKey: @"CFBundleTypeRole"] == NULL)
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] removeObjectForKey: @"CFBundleTypeRole"];
	if ([[appRoleRadioButtons selectedCell] tag] == 1)
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: @"Editor" forKey: @"CFBundleTypeRole"];
	else if ([[appRoleRadioButtons selectedCell] tag] == 2)
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: @"Viewer" forKey: @"CFBundleTypeRole"];
}


- (IBAction)addFileType:(id)sender
{
	NSMutableArray *typesArray, *mutableArray;
	int	selectedDocType = [docTypesDataBrowser selectedRow];
	
	if (selectedDocType == -1)
		return;

	typesArray = [[[[docTypeList array] objectAtIndex: selectedDocType ] dict] objectForKey: @"CFBundleTypeOSTypes"];
	if (typesArray == NULL)//if there is no file types array, we add it
	{
		typesArray = [[[NSMutableArray alloc] initWithCapacity: 255] retain]; 
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: typesArray forKey: @"CFBundleTypeOSTypes"];
	}
	else if ([typesArray isKindOfClass: [NSArray class]] && [typesArray count] == 0)//if it's not a mutable array, we make it mutable
	{
		mutableArray = [[[NSMutableArray alloc] initWithCapacity: 255] retain]; 
		[mutableArray addObjectsFromArray: typesArray];
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: mutableArray forKey: @"CFBundleTypeOSTypes"];
	}
	
	if (typesArray != NULL)//add new file type
	{
		[typesArray addObject: @"****"];
		[typeDataBrowser reloadData];
		[typeDataBrowser selectRow: [typesArray count]-1 byExtendingSelection: NO];
		[typeDataBrowser editColumn:0 row:([typesArray count]-1) withEvent:nil select:YES];
	}
}

- (IBAction)addSuffix:(id)sender
{
	NSMutableArray *suffixArray,*mutableArray;
	int	selectedDocType = [docTypesDataBrowser selectedRow];
	
	if (selectedDocType == -1)
		return;

	suffixArray = [[[[docTypeList array] objectAtIndex: selectedDocType ] dict] objectForKey: @"CFBundleTypeExtensions"];
	if (suffixArray == NULL)//if there is no suffix array, we add it
	{
		suffixArray = [[[NSMutableArray alloc] initWithCapacity: 255] retain]; 
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: suffixArray forKey: @"CFBundleTypeExtensions"];
	}
	else if ([suffixArray isKindOfClass: [NSArray class]] && [suffixArray count] == 0)//if it's not a mutable array, we make it mutable
	{
		mutableArray = [[[NSMutableArray alloc] initWithCapacity: 255] retain]; 
		[mutableArray addObjectsFromArray: suffixArray];
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] setObject: mutableArray forKey: @"CFBundleTypeExtensions"];
	}
	
	if (suffixArray != NULL)//add an item to the array
	{
		[suffixArray addObject: @"suffix"];
		[suffixesDataBrowser reloadData];
		[suffixesDataBrowser selectRow: [suffixArray count]-1 byExtendingSelection: NO];
		[suffixesDataBrowser editColumn:0 row:([suffixArray count]-1) withEvent:nil select:YES];
	}
}

- (IBAction)addDocType:(id)sender
{
	NSMutableDictionary		*docTypeDict = [[NSMutableDictionary dictionaryWithObjectsAndKeys: [NSMutableArray arrayWithObject: @"suffix"], @"CFBundleTypeExtensions",
																							  [NSMutableArray arrayWithObject: @"****"], @"CFBundleTypeOSTypes",
																							  @"Untitled Document Type", @"CFBundleTypeName",
																							  @"Viewer", @"CFBundleTypeRole", nil] retain];
																							  
		
	DocType *theType = [DocType docTypeWithDictionary: docTypeDict forBundle: appBundle];
	[[docTypeList array] addObject: theType];
	[docTypesDataBrowser reloadData];
	[docTypesDataBrowser selectRow: [[docTypeList array] count]-1 byExtendingSelection: NO];
	[docTypesDataBrowser editColumn:1 row:([[docTypeList array] count]-1) withEvent:nil select:YES];
	[self updateNumDocTypes];
}

- (IBAction)editDocType:(id)sender
{
	if ([docTypesDataBrowser selectedRow] != -1)
		[docTypesDataBrowser editColumn: 1 row: [docTypesDataBrowser selectedRow] withEvent: nil select: YES];
}

- (void)updateNumDocTypes
{
	int numTypes = [[docTypeList array] count];
	[numDocTypesTextField setStringValue: [NSString stringWithFormat: @"%d Document Types",numTypes]];
}


#pragma mark -

- (IBAction)appIconChanged:(id)sender
{
	[appIconView performClick: self];
}

- (IBAction)copyAppIcon:(id)sender
{
	[[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject: NSTIFFPboardType] owner: self];
	[[NSPasteboard generalPasteboard] setData: [[appIconView image] TIFFRepresentation] forType: NSTIFFPboardType];
}

- (IBAction)pasteAppIcon:(id)sender
{
	NSImage *pastedIcon = [[[NSImage alloc] initWithPasteboard: [NSPasteboard generalPasteboard]] autorelease];
	if (pastedIcon != NULL)
	{
		[appIconView setImage: pastedIcon];
		[self appIconChanged: self];
	}
}

- (IBAction)restoreApplicationIcon:(id)sender
{
	NSString	*iconPath = [appBundle pathForResource: [STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] ofType: @"icns"];
	if ([[NSFileManager defaultManager] fileExistsAtPath:iconPath isDirectory: nil])
	{ 
		NSImage		*appIcon = [[[NSImage alloc] initWithContentsOfFile: iconPath] autorelease];
		if (appIcon != NULL)
		{
			[appIconView setImage: appIcon];
			[iconNameTextField setStringValue: [iconPath lastPathComponent]];
			[self appIconChanged: self];
		}
	}
}

- (IBAction)exportAppIcon:(id)sender
{
	NSSavePanel *sPanel = [NSSavePanel savePanel];
	[sPanel setPrompt:@"Save"];
	[sPanel setTitle:@"Save Application Icon"];
	if ([sPanel runModalForDirectory:nil file: [iconNameTextField stringValue]] == NSOKButton) 
	{
		NSString *iconPath = [sPanel filename];
		IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [appIconView image]] autorelease];
		[iconFam writeToFile: iconPath];
	}
}


- (IBAction)importAppIcon:(id)sender
{
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setTitle: @"Import Icon For App"];
	[oPanel setPrompt:@"Import"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	//run open panel
    if ([oPanel runModalForDirectory:nil file:nil types:nil] == NSOKButton) 
	{
		IconFamily  *icon = [IconFamily iconFamilyWithIconOfFile: [oPanel filename]];
		if (icon != NULL)
		{
			[appIconView setImage: [icon imageWithAllReps]];
			[iconNameTextField setStringValue: [NSString stringWithFormat: @"%@ Icon", [[oPanel filename] lastPathComponent] ]];
			[self appIconChanged: self];
		}
		else
			[STUtil alert:@"Error getting icon" subText: @"The icon of the file you selected appears to be damaged or corrupt."];

	}
}

- (IBAction)revealAppIcon:(id)sender
{
	BOOL		isDir;

	if ([infoPlist objectForKey: @"CFBundleIconFile"] != NULL)
	{
		NSString	*path = [appBundle pathForResource: [STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] ofType: @"icns"];
		if ([[NSFileManager defaultManager] fileExistsAtPath:path isDirectory:&isDir]) 
			[[NSWorkspace sharedWorkspace] selectFile:path inFileViewerRootedAtPath:nil];
	}
}

- (IBAction)selectAppIcon:(id)sender
{
	//file types acceptable as custom icon data
	NSArray *fileTypes = [NSArray arrayWithObjects: @"icns", @"jpg", @"jpeg",@"gif",@"png",@"tif",@"tiff",@"bmp",@"pict",@"psd",@"pdf",@"tga",@"sgi",@"pntg", nil];

	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setTitle: @"Select Image For App"];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	//run open panel
    if ([oPanel runModalForDirectory:nil file:nil types:fileTypes] == NSOKButton) 
	{
		NSImage *customIcon = [[NSImage alloc] initWithContentsOfFile: [oPanel filename]];
		if (customIcon != NULL)
		{
			[appIconView setImage: customIcon];
			[customIcon autorelease];
			[iconNameTextField setStringValue: [NSString stringWithFormat: @"%@ Icon", [[oPanel filename] lastPathComponent]]];
			[self appIconChanged: self];
		}
		else
			[STUtil alert:@"Corrupt Image File" subText: @"The image file you selected appears to be damaged or corrupt."];
	}
}


#pragma mark -

- (IBAction)documentIconChanged:(id)sender
{
	int i;
	int	selectedDocType = [docTypesDataBrowser selectedRow];
	NSString *nameOfIconBeingChanged = [[[docTypeList array] objectAtIndex: selectedDocType] getIconName];
	
	for (i = 0; i < [[docTypeList array] count]; i++)
	{
		NSString *iconName = [[[docTypeList array] objectAtIndex: i] getIconName];
		if ([iconName isEqualToString: nameOfIconBeingChanged])
			[[[docTypeList array] objectAtIndex: i] setIcon: [documentIconView image]];
	}
	[[[docTypeList array] objectAtIndex: selectedDocType] setIcon: [documentIconView image]];
	[docTypesDataBrowser reloadData];
	[documentIconView performClick: self];
}

- (IBAction)copyDocumentIcon:(id)sender
{
	[[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject: NSTIFFPboardType] owner: self];
	[[NSPasteboard generalPasteboard] setData: [[documentIconView image] TIFFRepresentation] forType: NSTIFFPboardType];

}

- (IBAction)pasteDocumentIcon:(id)sender
{
	NSImage *pastedIcon = [[[NSImage alloc] initWithPasteboard: [NSPasteboard generalPasteboard]] autorelease];
	if (pastedIcon != NULL)
	{
		int	selectedDocType = [docTypesDataBrowser selectedRow];
		[[[docTypeList array] objectAtIndex: selectedDocType] setIcon: pastedIcon];
		[docTypesDataBrowser reloadData];
		[documentIconView setImage: pastedIcon];
		[self documentIconChanged: self];
	}
}

- (IBAction)restoreDocumentIcon:(id)sender
{	
	//get name of icon file
	NSString	*iconFileName = [[[[docTypeList array] objectAtIndex: [docTypesDataBrowser selectedRow]] dict] objectForKey: @"CFBundleTypeIconFile"];
	if (iconFileName == NULL)
		return;
	
	//get path to icon file in app bundle
	NSString	*iconPath = [appBundle pathForResource: [STUtil cutSuffix: iconFileName] ofType: @"icns"];
	if (![[NSFileManager defaultManager] fileExistsAtPath: iconPath isDirectory: nil])
	{
		[STUtil alert:@"No Icon Found" subText: [NSString stringWithFormat: @"The icon file %@ does not exist within the application bundle.", iconFileName]];
		return;
	}
	
	//create image from icon file and set it
	NSImage		*documentIcon = [[[NSImage alloc] initWithContentsOfFile: iconPath] autorelease];
	if (documentIcon != NULL)
	{
		[[[docTypeList array] objectAtIndex: [docTypesDataBrowser selectedRow]] setIcon: documentIcon];
		[documentIconView setImage: documentIcon];
		[docTypesDataBrowser reloadData];
		[documentIconNameTextField setStringValue: [iconPath lastPathComponent]];
		[self documentIconChanged: self];
	}
}

- (IBAction)exportDocumentIcon:(id)sender
{
	NSSavePanel *sPanel = [NSSavePanel savePanel];
	[sPanel setPrompt:@"Save"];
	[sPanel setTitle:@"Save Document Icon"];
	if ([sPanel runModalForDirectory:nil file: [documentIconNameTextField stringValue]] == NSOKButton) 
	{
		NSString *iconPath = [sPanel filename];
		IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [documentIconView image]] autorelease];
		[iconFam writeToFile: iconPath];
	}
}

- (IBAction)revealDocumentIcon:(id)sender
{
	BOOL		isDir;
	NSString	*iconFileName = [[[[docTypeList array] objectAtIndex: [docTypesDataBrowser selectedRow]] dict] objectForKey: @"CFBundleTypeIconFile"];
	
	if (iconFileName != NULL)
	{
		NSString	*path = [appBundle pathForResource: [STUtil cutSuffix: iconFileName] ofType: @"icns"];
		if ([[NSFileManager defaultManager] fileExistsAtPath:path isDirectory:&isDir]) 
			[[NSWorkspace sharedWorkspace] selectFile:path inFileViewerRootedAtPath:nil];
		else
			[STUtil alert:@"No Icon Found" subText: [NSString stringWithFormat: @"The icon file %@ does not exist within the application bundle.", iconFileName]];
	}
}


- (IBAction)importDocumentIcon:(id)sender
{
	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setTitle: @"Import Icon For Document"];
	[oPanel setPrompt:@"Import"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	//run open panel
    if ([oPanel runModalForDirectory:nil file:nil types:nil] == NSOKButton) 
	{
		IconFamily  *icon = [IconFamily iconFamilyWithIconOfFile: [oPanel filename]];
		if (icon != NULL)
		{
			[documentIconView setImage: [icon imageWithAllReps]];
			[self documentIconChanged: self];
		}
		else
			[STUtil alert:@"Error getting icon" subText: @"The icon of the file you selected appears to be damaged or corrupt."];

	}

}

- (IBAction)selectDocumentIcon:(id)sender
{
		//file types acceptable as custom icon data
	NSArray *fileTypes = [NSArray arrayWithObjects: @"icns", @"jpg", @"jpeg",@"gif",@"png",@"tif",@"tiff",@"bmp",@"pict",@"psd",@"pdf",@"tga",@"sgi",@"pntg", nil];

	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setTitle: @"Select Image For Document"];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	//run open panel
    if ([oPanel runModalForDirectory:nil file:nil types:fileTypes] == NSOKButton) 
	{
		NSImage *customIcon = [[NSImage alloc] initWithContentsOfFile: [oPanel filename]];
		if (customIcon != NULL)
		{
			[documentIconView setImage: customIcon];
			[customIcon autorelease];
			[self documentIconChanged: self];
		}
		else
			[STUtil alert:@"Corrupt Image File" subText: @"The image file you selected appears to be damaged or corrupt."];
	}
}

#pragma mark -

- (IBAction)removeFileType:(id)sender
{
	int selectedDocType = [docTypesDataBrowser selectedRow];
	NSMutableArray *fileTypesArray = [[[[docTypeList array] objectAtIndex: selectedDocType ] dict] objectForKey: @"CFBundleTypeOSTypes"];
	[fileTypesArray removeObjectAtIndex: [typeDataBrowser selectedRow]];
	if ([fileTypesArray count] == 0)//no objects
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] removeObjectForKey: @"CFBundleTypeOSTypes"];
	[typeDataBrowser reloadData];
}

- (IBAction)removeSuffix:(id)sender
{
	int selectedDocType = [docTypesDataBrowser selectedRow];
	NSMutableArray *suffixArray = [[[[docTypeList array] objectAtIndex: selectedDocType ] dict] objectForKey: @"CFBundleTypeExtensions"];
	[suffixArray removeObjectAtIndex: [suffixesDataBrowser selectedRow]];
	if ([suffixArray count] == 0)//no objects
		[[[[docTypeList array] objectAtIndex: selectedDocType ] dict] removeObjectForKey: @"CFBundleTypeExtensions"];
	[suffixesDataBrowser reloadData];
}

- (IBAction)removeDocType:(id)sender
{
	[[docTypeList array] removeObjectAtIndex: [docTypesDataBrowser selectedRow]];
	[docTypesDataBrowser reloadData];
	[self updateNumDocTypes];
}

- (IBAction)revert:(id)sender
{
	[self loadApp: [appBundle bundlePath]];
	[self tableViewSelectionDidChange: nil];
}

- (IBAction)save:(id)sender
{
	if ([[NSUserDefaults standardUserDefaults] boolForKey: @"safeMode"])
	{
		if ([STUtil proceedWarning: @"Are you sure you want to save?" subText: @"Are you sure you wish to proceed with applying the changes you have made?  These changes will not be reversible."])
			[self saveApp];
	}
}

- (IBAction)saveAs:(id)sender
{
	[appBundle bundlePath];
	
	NSSavePanel *sPanel = [NSSavePanel savePanel];
	[sPanel setPrompt:@"Save As.."];
	[sPanel setTitle:@"Save As..."];
	if ([sPanel runModalForDirectory:nil file: [window title]] == NSOKButton) 
	{
		NSString *newAppPath = [sPanel filename];
		//make a copy of the app in the new location
		[[NSFileManager defaultManager] copyPath: [appBundle bundlePath] toPath: newAppPath handler: nil];
		appBundle = [NSBundle bundleWithPath: newAppPath];
		[window setTitle: [newAppPath lastPathComponent]];
		if (infoPlistPath != NULL)
			[infoPlistPath release];
		if ([[NSFileManager defaultManager] fileExistsAtPath: [newAppPath stringByAppendingString: @"/Contents/Info.plist"] isDirectory: nil])
			infoPlistPath = [[newAppPath stringByAppendingString: @"/Contents/Info.plist"] retain];
		else if ([[NSFileManager defaultManager] fileExistsAtPath: [newAppPath stringByAppendingString: @"/Contents/Info-macos.plist"] isDirectory: nil])
			infoPlistPath = [[newAppPath stringByAppendingString: @"/Contents/Info-macos.plist"] retain];
		else
		{
			[STUtil alert: @"Incomplete Application Bundle" subText: [NSString stringWithFormat: @"No Info.plist file could be found within the application bundle at %@", newAppPath]];
			return;
		}
		[self saveApp];
	}
}


- (void)saveApp
{
	short				i;

	//make sure property list exists
	if(![[NSFileManager defaultManager] fileExistsAtPath: [appBundle bundlePath] isDirectory: nil])
	{
		[STUtil alert: @"Application not found" subText: [NSString stringWithFormat: @"The application bundle %@ could not be located and thus cannot be saved.",  [appBundle bundlePath]]];
		return;
	}

	//start saving by showing progress indicator
	[loadIndicator startAnimation: self];
	
	//update application icon
	NSData	*newAppIconData = [[appIconView image] TIFFRepresentation];
	NSString *appIconPath = [appBundle pathForResource: [STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] ofType: @"icns"];
	NSImage *oldAppIconImage = [[[NSImage alloc] initWithContentsOfFile: appIconPath] autorelease];
	NSData	*oldAppIconData = [oldAppIconImage TIFFRepresentation];
	if (![newAppIconData isEqualToData: oldAppIconData])//icon has changed
	{
		IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [appIconView image]] autorelease];
		[iconFam writeToFile: appIconPath];
	}
	
	//update document icons
	for (i = 0; i < [[docTypeList array] count]; i++)
	{
		DocType		*type = [[docTypeList array] objectAtIndex: i];
		if (![[type getIconName] isEqualToString: @"<Not Set>"]) //it DOES have a corresponding icns file in app bundle
		{
			NSString	*docIconPath = [appBundle pathForResource: [STUtil cutSuffix: [type getIconName]] ofType: @"icns"];
			//compare image in bundle with the image in the documentIconView
			NSData	*newDocIconData = [[documentIconView image] TIFFRepresentation];
			NSImage *oldDocIconImage = [[[NSImage alloc] initWithContentsOfFile: docIconPath] autorelease];
			NSData	*oldDocIconData = [oldDocIconImage TIFFRepresentation];
			if (![newDocIconData isEqualToData: oldDocIconData])//icon has changed
			{	//write new image to the icon file
				IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [type getIcon]] autorelease];
				[iconFam writeToFile: docIconPath];
			}
		}
		else	//ok, there is no corresponding icon file in bundle. 
		{
			NSData	*noIconIconData = [[NSImage imageNamed: @"noicon"] TIFFRepresentation];
			NSData	*newDocIconData = [[documentIconView image] TIFFRepresentation];
			//an icon has been set for this formerly iconless doc type
			if (![newDocIconData isEqualToData: noIconIconData]) 
			{
				//create the new icns file
				NSString *iconFileName = [NSString stringWithFormat: @"%@.icns", [type getDocTypeName]];
				NSString *docIconPath = [NSString stringWithFormat: @"%@/%@", [appBundle resourcePath], iconFileName];
				//write new image to the icon file
				IconFamily *iconFam = [[[IconFamily alloc] initWithThumbnailsOfImage: [type getIcon]] autorelease];
				[iconFam writeToFile: docIconPath];
				//update the dictionary
				[type setIconName: iconFileName];
			}
		}
	}
	
	
	
	//write property list to app bundle
 	NSMutableDictionary	*newPlist = [self generatePropertyListFromSettings];
	[newPlist writeToFile: infoPlistPath atomically: YES];
	
	//update
	[self tableViewSelectionDidChange: nil];
	
	//stop progress indicator since we're done saving
	[loadIndicator stopAnimation: self];
}



- (NSMutableDictionary *)generatePropertyListFromSettings
{
	NSMutableDictionary *newPlist = [NSMutableDictionary dictionaryWithCapacity: 255];
	NSMutableArray		*docTypesArray= [NSMutableArray arrayWithCapacity: 255];
	short				i;
	
	//load the original property list, to begin with
	[newPlist addEntriesFromDictionary: infoPlist];
	
//set bundle strings
	// bundle name
	if ([[bundleNameTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleName"];
	else
		[newPlist setObject: [bundleNameTextField stringValue] forKey: @"CFBundleName"];
	// bundle identifier
	if ([[bundleIdentifierTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleIdentifier"];
	else
		[newPlist setObject: [bundleIdentifierTextField stringValue] forKey: @"CFBundleIdentifier"];
	// bundle signature
	if ([[bundleSignatureTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleSignature"];
	else
		[newPlist setObject: [bundleSignatureTextField stringValue] forKey: @"CFBundleSignature"];
	// bundle package type
	if ([[bundlePackageTypeTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundlePackageType"];
	else
		[newPlist setObject: [bundlePackageTypeTextField stringValue] forKey: @"CFBundlePackageType"];
	// bundle version
	if ([[bundleVersionTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleVersion"];
	else
		[newPlist setObject: [bundleVersionTextField stringValue] forKey: @"CFBundleVersion"];
	//bundle short version string
	if ([[bundleShortVersionTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleShortVersionString"];
	else
		[newPlist setObject: [bundleShortVersionTextField stringValue] forKey: @"CFBundleShortVersionString"];
	// bundle get info string
	if ([[bundleGetInfoTextField stringValue] length] == 0)
		[newPlist removeObjectForKey: @"CFBundleGetInfoString"];
	else
		[newPlist setObject: [bundleGetInfoTextField stringValue] forKey: @"CFBundleGetInfoString"];
	
	//generate CFBundleDocumentTypes array
	for (i = 0; i < [[docTypeList array] count]; i++)
	{
		NSMutableDictionary	*entry = [[[docTypeList array] objectAtIndex: i] dict];
		
		//if there is an empty extensions or types array, we remove it -- no point in its presence in the plist
		if ([entry objectForKey: @"CFBundleTypeExtensions"] != NULL && [[entry objectForKey: @"CFBundleTypeExtensions"] count] == 0)
			[entry removeObjectForKey: @"CFBundleTypeExtensions"];
		if ([entry objectForKey: @"CFBundleTypeOSTypes"] != NULL && [[entry objectForKey: @"CFBundleTypeOSTypes"] count] == 0)
			[entry removeObjectForKey: @"CFBundleTypeOSTypes"];
		
		[docTypesArray addObject: entry];
	}
	
	if ([docTypesArray count] == 0 && [newPlist objectForKey: @"CFBundleDocumentTypes"] != NULL)
		[newPlist removeObjectForKey: @"CFBundleDocumentTypes"];
	else
		[newPlist setObject: docTypesArray forKey: @"CFBundleDocumentTypes"];
	
	return newPlist;
}



- (void)loadApp: (NSString *)appPath
{
	short i;
	
	//make sure there is actually something at the path
	if(![[NSFileManager defaultManager] fileExistsAtPath: appPath isDirectory: nil])
	{
		[STUtil alert: @"No Such Application" subText: [NSString stringWithFormat: @"No application bundle exists at the path %@", appPath]];
		return;
	}
	//make sure it's an application package, not an old-fashioned Carbon app
	if (![[NSWorkspace sharedWorkspace] isFilePackageAtPath: appPath])
	{
		[STUtil alert: @"Not an Application Package" subText: [NSString stringWithFormat: @"The application you selected is not a Mac OS X application package.", appPath]];
		return;
	}

	//set window title
	[window setTitle: [appPath lastPathComponent]];

	//get bundle
	appBundle = [NSBundle bundleWithPath: appPath];
	
	//make sure property list exists
	if (infoPlistPath != NULL)
		[infoPlistPath release];
	if ([[NSFileManager defaultManager] fileExistsAtPath: [appPath stringByAppendingString: @"/Contents/Info.plist"] isDirectory: nil])
		infoPlistPath = [[appPath stringByAppendingString: @"/Contents/Info.plist"] retain];
	else if ([[NSFileManager defaultManager] fileExistsAtPath: [appPath stringByAppendingString: @"/Contents/Info-macos.plist"] isDirectory: nil])
		infoPlistPath = [[appPath stringByAppendingString: @"/Contents/Info-macos.plist"] retain];
	else
	{
		[STUtil alert: @"Incomplete Application Bundle" subText: [NSString stringWithFormat: @"No Info.plist file could be found within the application bundle", appPath]];
		return;
	}
	//load property list
	infoPlist = [[NSDictionary dictionaryWithContentsOfFile: infoPlistPath] retain];

	//set icon
	if ([infoPlist objectForKey: @"CFBundleIconFile"] != NULL)
	{
		NSString	*iconPath = [appBundle pathForResource: [STUtil cutSuffix: [infoPlist objectForKey: @"CFBundleIconFile"]] ofType: @"icns"];
		NSImage		*appIcon = [[[NSImage alloc] initWithContentsOfFile: iconPath] autorelease];
		[appIconView setImage: appIcon];
		[iconNameTextField setStringValue: [iconPath lastPathComponent]];
		[exportAppIconButton setEnabled: YES];
		[revealAppIconButton setEnabled: YES];
	}
	else
	{
		NSImage		*appIcon = [NSImage imageNamed: @"GenericApplicationIcon"];
		[appIconView setImage: appIcon];
		[iconNameTextField setStringValue: @"<Not Set>"];
		[exportAppIconButton setEnabled: NO];
		[revealAppIconButton setEnabled: NO];
	}
	
	//set bundle strings
	if ([infoPlist objectForKey: @"CFBundleName"] != NULL)
		[bundleNameTextField setStringValue: [infoPlist objectForKey: @"CFBundleName"]];
	if ([infoPlist objectForKey: @"CFBundleIdentifier"] != NULL)
		[bundleIdentifierTextField setStringValue: [infoPlist objectForKey: @"CFBundleIdentifier"]];
	if ([infoPlist objectForKey: @"CFBundleSignature"] != NULL)
		[bundleSignatureTextField setStringValue: [infoPlist objectForKey: @"CFBundleSignature"]];
	if ([infoPlist objectForKey: @"CFBundlePackageType"] != NULL)
		[bundlePackageTypeTextField setStringValue: [infoPlist objectForKey: @"CFBundlePackageType"]];
	if ([infoPlist objectForKey: @"CFBundleVersion"] != NULL)
		[bundleVersionTextField setStringValue: [infoPlist objectForKey: @"CFBundleVersion"]];
	if ([infoPlist objectForKey: @"CFBundleShortVersionString"] != NULL)
		[bundleShortVersionTextField setStringValue: [infoPlist objectForKey: @"CFBundleShortVersionString"]];
	if ([infoPlist objectForKey: @"CFBundleGetInfoString"] != NULL)
		[bundleGetInfoTextField setStringValue: [infoPlist objectForKey: @"CFBundleGetInfoString"]];

	//disable certain controls while still loading
	[addDocTypeButton setEnabled: NO];
	
	
	//show window
	[docTypesDataBrowser deselectAll: self];
	[docTypesDataBrowser setDataSource: nil];
	if (![window isVisible])
		[window center];
	[window makeKeyAndOrderFront: self];

	//set load indicator off
	[loadIndicator setUsesThreadedAnimation: YES];
	[loadIndicator startAnimation: self];

	//create DocTypeList and set as data source
	if (docTypeList != NULL) { [docTypeList release]; }
	docTypeList = [[DocTypeList alloc] init];
	NSArray *docDictArray = [infoPlist objectForKey: @"CFBundleDocumentTypes"];
	if (docDictArray != NULL)
	{
		for (i = 0; i < [docDictArray count]; i++)
		{
			DocType *theType = [DocType docTypeWithDictionary: [docDictArray objectAtIndex: i] forBundle: appBundle];
			[[docTypeList array] addObject: theType];
		}
	}
	[docTypesDataBrowser setDataSource: docTypeList];
	[docTypesDataBrowser reloadData];
	[self updateNumDocTypes];
	
	//create Nib list
	nibList = [[NibList alloc] initWithResourcesPath: [appBundle resourcePath]];
	[nibListDataBrowser setDataSource: nibList];
	[nibListDataBrowser reloadData];
	
	//enable the controls we disabled
	[addDocTypeButton setEnabled: YES];
	
	[loadIndicator stopAnimation: self];
}

- (IBAction)revealNib:(id)sender
{
	NSString *nibPath = [[nibListDataBrowser itemAtRow: [nibListDataBrowser selectedRow]] objectForKey: @"Path"];
	if (nibPath != NULL)
	{
		if ([[NSFileManager defaultManager] fileExistsAtPath: nibPath isDirectory: nil]) 
			[[NSWorkspace sharedWorkspace] selectFile: nibPath inFileViewerRootedAtPath:nil];
	}
}

- (IBAction)editNib:(id)sender
{
	NSString *nibPath = [[nibListDataBrowser itemAtRow: [nibListDataBrowser selectedRow]] objectForKey: @"Path"];
	if (nibPath != NULL)
	{
		if ([[NSFileManager defaultManager] fileExistsAtPath: nibPath isDirectory: nil]) 
			[[NSWorkspace sharedWorkspace] openFile: nibPath withApplication: @"Interface Builder" andDeactivate: YES];
	}
}

// Delegate to disallow editing of cells
- (BOOL)outlineView:(NSOutlineView *)outlineView shouldEditTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
	return NO;
}

// Delegate to manage changing selection of outline view in nib drawer
- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
	int selected = [nibListDataBrowser selectedRow];

	if (selected == -1)//nothing selected
	{
		//show "Empty Selection" box
		[emptyNibSelectionBox setHidden: NO];
		//hide all the nib controls
		[nibNameTextField setHidden: YES];
		[nibIconView setHidden: YES];
		[revealNibButton setHidden: YES];
		[nibEditButton setHidden: YES];
	}
	else
	{
		id	item = [nibListDataBrowser itemAtRow: selected];
		if (item != NULL)
		{
			[nibIconView setImage: [[NSWorkspace sharedWorkspace] iconForFile: [item objectForKey: @"Path"]]];
			[emptyNibSelectionBox setHidden: YES];
			[nibIconView setHidden: NO];
			[revealNibButton setHidden: NO];
			// No edit button for items which aren't Nibz
			if ([[item objectForKey: @"Name"] hasSuffix: @"nib"])
				[nibEditButton setHidden: NO];
			else
				[nibEditButton setHidden: YES];
			[nibNameTextField setHidden: NO];
			[nibNameTextField setStringValue: [item objectForKey: @"Name"]];
		}
	}
}


/*****************************************
 - Delegate managing selection in the Bundled Files list
*****************************************/

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
	int selected, selectedDocType;
	
	if ([aNotification object] == docTypesDataBrowser || [aNotification object] == NULL)
	{
		selected = [docTypesDataBrowser selectedRow];
		if (selected != -1) //there is a selected item
		{
			//show document info box
			[documentBox setHidden: NO];
			[noDocumentBox setHidden: YES];
		
			//update icon in Document Type View
			[documentIconView setImage: [[[docTypeList array] objectAtIndex: selected] getIcon]];
			
			//suffix list
			[suffixesDataBrowser setDataSource: [[docTypeList array] objectAtIndex: selected]];
			[suffixesDataBrowser reloadData];
			//type list
			[typeDataBrowser setDataSource: [[docTypeList array] objectAtIndex: selected]];
			[typeDataBrowser reloadData];
			
			//set app role
			if ([[[[docTypeList array] objectAtIndex: selected] dict] objectForKey: @"CFBundleTypeRole"] == NULL || [(NSString *)[[[[docTypeList array] objectAtIndex: selected] dict] objectForKey: @"CFBundleTypeRole"] length] == 0)
				[appRoleRadioButtons selectCellWithTag: 0];
			else if ([[[[[docTypeList array] objectAtIndex: selected] dict] objectForKey: @"CFBundleTypeRole"] isEqualToString: @"Editor"])
				[appRoleRadioButtons selectCellWithTag: 1];
			else if ([[[[[docTypeList array] objectAtIndex: selected] dict] objectForKey: @"CFBundleTypeRole"] isEqualToString: @"Viewer"])
				[appRoleRadioButtons selectCellWithTag: 2];
			else
				[appRoleRadioButtons selectCellWithTag: 0];
			
			//set icon name
			[documentIconNameTextField setStringValue: [[[docTypeList array] objectAtIndex: selected] getIconName]];
			
			//enable "Remove Type" button
			[removeDocTypeButton setEnabled: YES];
			
			//set title of document box
			[documentBox setTitle: [[[docTypeList array] objectAtIndex: selected] getDocTypeName]];
			
			//enable/disable buttons according to whether this is a real icon or just the empty placeholder
			if ([[documentIconNameTextField stringValue] isEqualToString: @"<Not Set>"])
			{
				[exportDocIconButton setEnabled: NO];
				[revealDocIconButton setEnabled: NO];
				[restoreDocIconButton setEnabled: NO];
			}
			else
			{
				[exportDocIconButton setEnabled: YES];
				[revealDocIconButton setEnabled: YES];
				[restoreDocIconButton setEnabled: YES];
			}
		}
		else
		{
			//disable remove type button
			[removeDocTypeButton setEnabled: NO];
			//set title of document box
			[documentBox setHidden: YES];
			[noDocumentBox setHidden: NO];
		}
	}
	else if ([aNotification object] == suffixesDataBrowser)
	{
		selected = [suffixesDataBrowser selectedRow];
			
		if (selected != -1) //there is a selected item
			[removeSuffixButton setEnabled: YES];
		else
			[removeSuffixButton setEnabled: NO];
	}
	else if ([aNotification object] == typeDataBrowser)
	{
		selected = [typeDataBrowser selectedRow];
			
		if (selected != -1) //there is a selected item
			[removeFileTypeButton setEnabled: YES];
		else
			[removeFileTypeButton setEnabled: NO];
	}
}

/*****************************************
 - Delegate for when text changes in any of the AppHack text fields
*****************************************/
- (void)controlTextDidChange:(NSNotification *)aNotification
{
	BOOL	isDir, exists = NO, validName = NO;
	
	//app name or script path was changed
	if ([aNotification object] == bundleSignatureTextField || [aNotification object] == bundlePackageTypeTextField)
	{
		NSRange	 range = { 0, 4 };
		NSString *sig = [[aNotification object] stringValue];
		
		if ([sig length] > 4)
		{
			[[aNotification object] setStringValue: [sig substringWithRange: range]];
		}
	}
}

/*****************************************
 - Delegate for enabling and disabling menu items
*****************************************/
- (BOOL)validateMenuItem:(NSMenuItem*)anItem 
{
	if (([[[anItem menu] title] isEqualToString: @"Bundle"]) && (![window isVisible]))
		return NO;
	
	//Documents Menu
	if (([[[anItem menu] title] isEqualToString: @"Documents"]))
	{
		if (![window isVisible])
			return NO;
		if ([[anItem title] isEqualToString:@"Add Document Type"])
			return YES;
		if ([docTypesDataBrowser selectedRow] == -1)
			return NO;
	}
	//View Menu
//	if ([[[anItem menu] title] isEqualToString: @"View"] && ![window isVisible])
//		return NO;
		
	if (([[anItem title] isEqualToString: @"Export App Icon"] ||
		 [[anItem title] isEqualToString: @"Reveal App Icon"])
		 && ![exportAppIconButton isEnabled])
		return NO;

	// disable Save (As) and Revert items if window isn't visible
	if (![window isVisible] && [[anItem title] isEqualToString: @"Save"])
		return NO;
	if (![window isVisible] && [[anItem title] isEqualToString: @"Save As"])
		return NO;
	if (![window isVisible] && [[anItem title] isEqualToString: @"Revert"])
		return NO;

	return YES;
}

#pragma mark -

- (NSToolbarItem *)toolbar:(NSToolbar *)theToolbar itemForItemIdentifier:(NSString *)itemIdentifier willBeInsertedIntoToolbar:(BOOL)flag
{
	NSToolbarItem	*item = [[[NSToolbarItem alloc] initWithItemIdentifier: itemIdentifier] autorelease];
	
	if ([itemIdentifier isEqual: NSToolbarCustomizeToolbarItemIdentifier]) 
	{
		[item setTarget: theToolbar];
		[item setAction:@selector(runCustomizationPalette:)];
	}
	if ([itemIdentifier isEqual: kShowNibDrawerToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"nibs"]];
		[item setLabel: @"Nibs"];
		[item setPaletteLabel: @"Nibs"];
		[item setToolTip: @"Toggled Nib Drawer"];
		[item setTarget: nibDrawer];
		[item setAction:@selector(toggle:)];
	}
	else if ([itemIdentifier isEqual: kExportAllIconsToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"exporticonset"]];
		[item setLabel: @"Export Icons"];
		[item setPaletteLabel: @"Export Icons"];
		[item setToolTip: @"Export Application Icon Set"];
		[item setTarget:self];
		[item setAction:@selector(exportAllIcons:)];
	}
	else if ([itemIdentifier isEqual: kImportIconSetToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"importiconset"]];
		[item setLabel: @"Import Icons"];
		[item setPaletteLabel: @"Import Icons"];
		[item setToolTip: @"Import an Icon Set"];
		[item setTarget:self];
		[item setAction:@selector(importIconSet:)];
	}
	else if ([itemIdentifier isEqual: kRevealApplicationToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"revealapp"]];
		[item setLabel: @"Reveal"];
		[item setPaletteLabel: @"Reveal"];
		[item setToolTip: @"Reveal Application In Finder"];
		[item setTarget:self];
		[item setAction:@selector(revealApplication:)];
	}
	else if ([itemIdentifier isEqual: kExportPlistToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"editplist"]];
		[item setLabel: @"Export Plist"];
		[item setPaletteLabel: @"Export Plist"];
		[item setToolTip: @"Export App Property List File"];
		[item setTarget:self];
		[item setAction:@selector(exportBundlePlist:)];
	}
	else if ([itemIdentifier isEqual: kImportPlistToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"editplist"]];
		[item setLabel: @"Import Plist"];
		[item setPaletteLabel: @"Import Plist"];
		[item setToolTip: @"Import a Property List File For App"];
		[item setTarget:self];
		[item setAction:@selector(importBundlePlist:)];
	}
	else if ([itemIdentifier isEqual: kShowContentsToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"showpackagecontents"]];
		[item setLabel: @"Contents"];
		[item setPaletteLabel: @"Contents"];
		[item setToolTip: @"Show Package Contents"];
		[item setTarget:self];
		[item setAction:@selector(showPackageContents:)];
	}
	else if ([itemIdentifier isEqual: kGetInfoToolbarItemIdentifier])
	{
		[item setImage: [NSImage imageNamed: @"finder"]];
		[item setLabel: @"Get Info"];
		[item setPaletteLabel: @"Get Info"];
		[item setToolTip: @"Get Finder Info on Application bundle"];
		[item setTarget:self];
		[item setAction:@selector(getFileInfo:)];
	}
	else if ([itemIdentifier isEqual: kSaveAppToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"savetoolbar"]];
		[item setLabel: @"Save"];
		[item setPaletteLabel: @"Save"];
		[item setToolTip: @"Save Changes To Application"];
		[item setTarget:self];
		[item setAction:@selector(save:)];
	}
	else if ([itemIdentifier isEqual: kRevertToolbarItemIdentifier]) 
	{
		[item setImage: [NSImage imageNamed: @"reverttoolbar"]];
		[item setLabel: @"Revert"];
		[item setPaletteLabel: @"Revert"];
		[item setToolTip: @"Revert"];
		[item setTarget:self];
		[item setAction:@selector(revert:)];
	}
	return (item);
}

- (NSArray *)toolbarAllowedItemIdentifiers:(NSToolbar *)toolbar
{
	NSArray	*itemIds = [NSArray arrayWithObjects: kShowNibDrawerToolbarItemIdentifier, 
												  kExportAllIconsToolbarItemIdentifier,
												  kImportIconSetToolbarItemIdentifier,
												  kRevealApplicationToolbarItemIdentifier,
												  kExportPlistToolbarItemIdentifier,
												  kImportPlistToolbarItemIdentifier,
												  kShowContentsToolbarItemIdentifier,
												  kRevertToolbarItemIdentifier,
												  kSaveAppToolbarItemIdentifier,
												  kGetInfoToolbarItemIdentifier,
												  NSToolbarFlexibleSpaceItemIdentifier,
												  NSToolbarSpaceItemIdentifier,
												  NSToolbarSeparatorItemIdentifier, 
												  NSToolbarCustomizeToolbarItemIdentifier, nil];
	return itemIds;
}

- (NSArray *)toolbarDefaultItemIdentifiers:(NSToolbar *)toolbar
{
	NSArray	*itemIds = [NSArray arrayWithObjects:	kRevealApplicationToolbarItemIdentifier,
													kShowContentsToolbarItemIdentifier,
													kGetInfoToolbarItemIdentifier,
													NSToolbarSeparatorItemIdentifier,
													kSaveAppToolbarItemIdentifier,
													kRevertToolbarItemIdentifier,
													NSToolbarSeparatorItemIdentifier,
													NSToolbarFlexibleSpaceItemIdentifier,
													NSToolbarSeparatorItemIdentifier,
													NSToolbarCustomizeToolbarItemIdentifier,
													kShowNibDrawerToolbarItemIdentifier, nil];
	return itemIds;
}
@end
