//
//  WANController.m
//  BwanaDik
//
//  Created by John Schilling on 9/8/06.
//  Copyright 2006 John Schilling. All rights reserved.
//

#import "WANController.h"
#import "Downloader.h"
#import "NSStringExtras.h"


#define kDefaultTimeout 20.0


@implementation WANController


#pragma mark
#pragma mark PRIVATE INSTANCE METHODS
#pragma mark


- (id)init
{
    if( ![super init] ) return nil;
    
	_currentStatus      = BDUnchecked;
        _previousStatus     = BDUnchecked;
        _currentAddress     = [[NSMutableString alloc] initWithString:@""];
        _previousAddress    = [[NSMutableString alloc] initWithString:@""];
        
        _IPsourceTag        = 0;
        _IPURL              = [[NSMutableString alloc] init];
        _IPTagStart         = [[NSMutableString alloc] init];
        _IPTagEnd           = [[NSMutableString alloc] init];
        
        _downloadTimeout    = kDefaultTimeout;
        
        _checking           = NO; //NOTE: Checking IS NOT the same as downloading. Checking is finsihed before download is finished,
                                  // and must be set manually, versus downloading, which just checks the status of the downloader.
        
        [self initDownloader];
        
    return self;
}

- (void)awakeFromNib
{

}

- (void)dealloc
{
    [_currentAddress    release];
    [_previousAddress   release];
    
    [_IPURL             release];
    [_IPTagStart        release];
    [_IPTagEnd          release];
    
    if (_downloader)    [_downloader release];
    _downloader = nil;
    
    [super dealloc];
}

- (void)cleanup
{
    [_downloader cleanup];
}




#pragma mark
#pragma mark NSURLConnection (downloader) control methods
#pragma mark

- (void)setDelegate:(id)delegate
{
    _delegate = delegate;
}

- (void)setLastKnownIPAddress:(NSString *)address
                downloadTimeout:(float)timeout
                IPSourceTag:(int)tag
{
    [self setLastKnownIPAddress:address];
    [self setDownloadTimeout:timeout];
    [self setIPSourceTag:tag];
}

- (void)setDownloadTimeout:(float)timeout
{
    _downloadTimeout = timeout;
}

- (void)setIPSourceTag:(int)tag
{
    _IPsourceTag = tag;
    [self resetDownloader];
}

- (void)setLastKnownIPAddress:(NSString *)address
{
    [_previousAddress setString:address];
}

- (void)setIPSourceStrings
{
    switch (_IPsourceTag) {
    
        case 0: // checkip.dyndns.org
            [_IPURL      setString:@"http://checkip.dyndns.org/"];
            [_IPTagStart setString:@"Current IP Address:"];
            [_IPTagEnd   setString:@"</body>"];
        break;
        
        case 1: // www.whatismyip.com
            [_IPURL      setString:@"http://www.whatismyip.com/"];
            [_IPTagStart setString:@"displaycopy('"];
            [_IPTagEnd   setString:@"');"];
        break;
        
        default: // checkip.dyndns.org
            [_IPURL      setString:@"http://checkip.dyndns.org/"];
            [_IPTagStart setString:@"Current IP Address:"];
            [_IPTagEnd   setString:@"</body>"];
        break;
    }
}

- (void)initDownloader
{
    [self setIPSourceStrings];
    _downloader = [[Downloader alloc] initWithURL:_IPURL delegate:self];
}

- (void)resetDownloader
{
    if (!_downloader) {
        [self initDownloader];
        return;
    }
    [self setIPSourceStrings];
    [_downloader setURL:_IPURL];
}





#pragma mark
#pragma mark ACCESS TO STATE VARIABLES
#pragma mark


- (NSString *)currentWANAddress
{
    return _currentAddress;
}

- (NSString *)previousWANAddress
{
    return _previousAddress;
}

- (BDConnectionState)currentOnlineStatus
{
    return _currentStatus;
}

- (BDConnectionState)previousOnlineStatus
{
    return _previousStatus;
}







#pragma mark
#pragma mark CORE WAN FETCHING PARSING METHODS
#pragma mark


- (void)parseAddressData:(NSData *)data
{
    NSString *ipTag;
    NSString *htmlString = [[NSString alloc] initWithBytes:[data bytes] length:[data length] encoding:NSASCIIStringEncoding];
    if (htmlString && [htmlString length] > 0) {
        NSScanner *ipScanner = [[NSScanner scannerWithString:htmlString] retain];
        [ipScanner scanUpToString:_IPTagStart intoString: nil];
        [ipScanner scanString:_IPTagStart intoString: nil];
        [ipScanner scanUpToString:_IPTagEnd intoString:&ipTag];
        if (ipTag && [ipTag length] > 0) {
            _currentStatus = BDConnected;
            [_currentAddress setString:[ipTag stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceAndNewlineCharacterSet]]];
         } else {
            _currentStatus = BDBusyServer;
            [_currentAddress setString:@""];
        }
        [ipScanner release];
        ipScanner = nil;
    } else {
        _currentStatus = BDUnknown;
        [_currentAddress setString:@""];
    }
    [htmlString release];
    htmlString = nil;
    
    [self downloadDidFinish];
    
    [self checkStatusChanged];
    [self checkAddressChanged];
    
    #ifdef __DEBUG__
    NSLog(@"IP Address: %@", _currentAddress);
    #endif
}

- (void)downloadDidFail
{
    // fail is like the opposite of parseAddressData. It is called when we arent connected. dont alert the delegate, just do as
    // parseAddressData does, downloadDidFinish and checkStatusChanged will alert the delegate.
    
    _currentStatus = BDDisconnected;
    [_currentAddress setString:@""];
    
    [self downloadDidFinish];
    [self checkStatusChanged];
}


- (void)downloadDidBegin
{
    if ([_delegate respondsToSelector:@selector(WANCheckBegun)]) [_delegate WANCheckBegun];
}

- (void)downloadDidCancel
{
    _checking = NO;
    [_currentAddress setString:@""];
    _currentStatus = 0;
    if ([_delegate respondsToSelector:@selector(WANCheckCanceled)]) [_delegate WANCheckCanceled];
}

- (void)downloadDidFinish
{
    _checking = NO;
    if ([_delegate respondsToSelector:@selector(WANCheckFinished)]) [_delegate WANCheckFinished];
}



- (void)checkStatusChanged
{
    if (_currentStatus != _previousStatus)
    {
        
        // UNCHECKED to BAD/UNKNOWN (DOWN)
        if (_previousStatus == BDUnchecked && _currentStatus == BDDisconnected) {
            // network went from unchecked to BAD.
            if ([_delegate respondsToSelector:@selector(WANWentOffline)]) [_delegate WANWentOffline];
        } else if (_previousStatus == BDUnchecked && _currentStatus == BDUnknown) {
            // network went from unchecked to UNKNOWN.
            if ([_delegate respondsToSelector:@selector(WANWentOffline)]) [_delegate WANWentOffline];
        }
    
        // BAD/UNKNWON/BUSY to GOOD (UP)
          else if ( _previousStatus == BDUnknown && _currentStatus == BDConnected) {
            // network went from unknown to good.
            if ([_delegate respondsToSelector:@selector(WANWentOnline)]) [_delegate WANWentOnline];
        } else if (_previousStatus == BDBusyServer && _currentStatus == BDConnected) {
            // network went from busy to good.
            if ([_delegate respondsToSelector:@selector(WANWentOnline)]) [_delegate WANWentOnline];
        } else if ( _previousStatus == BDDisconnected && _currentStatus == BDConnected) {
            // network went from bad to good.
            if ([_delegate respondsToSelector:@selector(WANWentOnline)]) [_delegate WANWentOnline];
        }
        
        // GOOD to BAD/UNKNOWN/BUSY (DOWN)
          else if (_previousStatus == BDConnected && _currentStatus == BDDisconnected) {
            // network went from good to bad
            if ([_delegate respondsToSelector:@selector(WANWentOffline)]) [_delegate WANWentOffline];
        } else if (_previousStatus == BDConnected && _currentStatus == BDBusyServer) {
            // network went from good to busy
            if ([_delegate respondsToSelector:@selector(WANWentOffline)]) [_delegate WANWentOffline];
        } else if (_previousStatus == BDConnected && _currentStatus == BDUnknown) {
            // network went from good to unknown
            if ([_delegate respondsToSelector:@selector(WANWentOffline)]) [_delegate WANWentOffline];
        }
        
        // BUSY/UNKNOWN/BAD to BUSY/UNKNOWN/BAD (technically, unchanged)
          else if (_previousStatus == BDUnknown && _currentStatus == BDBusyServer) {
            // network went from unknown to busy
            //!!DELEGATE NOTIFIED OF CHANGE
        } else if (_previousStatus == BDUnknown && _currentStatus == BDDisconnected) {
            // network went from unknown to bad
            //!!DELEGATE NOTIFIED OF CHANGE
        }
        
          else if (_previousStatus == BDDisconnected && _currentStatus == BDUnknown) {
            // network went from bad to unknown
            //!!DELEGATE NOTIFIED OF CHANGE
        } else if (_previousStatus == BDDisconnected && _currentStatus == BDBusyServer) {
            // network went from bad to busy
            //!!DELEGATE NOTIFIED OF CHANGE
        } 
        
          else if (_previousStatus == BDBusyServer && _currentStatus == BDUnknown) {
            // network went from busy to unknown
            //!!DELEGATE NOTIFIED OF CHANGE
        } else if (_previousStatus == BDBusyServer && _currentStatus == BDDisconnected) {
            // network went from busy to bad
            //!!DELEGATE NOTIFIED OF CHANGE
        }
        
        
        #ifdef __DEBUG__
        NSLog(@"_currentOnlineStatus == %d, _previousOnlineStatus == %d", _currentStatus, _previousStatus);
        #endif
    }
    _previousStatus = _currentStatus;
}

- (void)checkAddressChanged
{
    // Okay, if the LAN changed, update the interface and get new WAN:
    if ([_currentAddress isValidAddress] && ![_currentAddress isEqualToString:_previousAddress])
    {
        // NOTE: only notify and change previosAddress if current is a valid IP address!
        
        if ([_previousAddress isValidAddress]) {
            // only notify if previous address had been set
            if ([_delegate respondsToSelector:@selector(WANAddressChangedFrom:to:)])
            {
                [_delegate WANAddressChangedFrom:_previousAddress to:_currentAddress];
            }
        }
        [_previousAddress setString:_currentAddress];
        #ifdef __DEBUG__
            //NSLog(@"WAN changed");
        #endif
    } else {
        #ifdef __DEBUG__
            //NSLog(@"WAN same or unknown");
        #endif
    }
}













#pragma mark
#pragma mark DOWNLOADER METHODS
#pragma mark

- (IBAction)beginChecking:(id)sender
{
    if (![self isDownloading]) {
        _checking = YES;
        [_downloader downloadDataWithTimeoutInterval:_downloadTimeout];
    }
}

- (IBAction)cancelChecking:(id)sender
{
    if (![self isDownloading]) return;
    [_downloader cancelDownloading];
    [self downloadDidCancel];
}

- (IBAction)checkOrCancel:(id)sender
{
    if ([self isDownloading]) {
        [self cancelChecking:nil];
    } else {
        [self beginChecking:nil];
    }
}

- (BOOL)isChecking
{
    return _checking;
    // NOTE: checking IS NOT THE SAME as downloading. We have finished checking as soon as the data comes in, not when the downloader has finished.
}

- (BOOL)isDownloading
{
    if (!_downloader) return NO;
    if ([_downloader isDownloading]) return YES;
    return NO;
}



#pragma mark
#pragma mark DOWNLOADER DELEGATE METHODS
#pragma mark

- (void)downloaderDidBegin:(Downloader *)downloader
{
    [self downloadDidBegin];
    #ifdef __DEBUG__
        NSLog(@"downloader: checking...");
    #endif
}

- (void)downloaderDidConnect:(Downloader *)downloader expectedLength:(int)expectedLength
{
    #ifdef __DEBUG__
        NSLog(@"downloader: connected...");
    #endif
}

- (void)downloaderDidFailWithError:(Downloader *)downloader error:(NSString *)error
{
    #ifdef __DEBUG__
        NSLog(error);
    #endif
    [self downloadDidFail];
}

- (void)downloaderDidReceiveData:(Downloader *)downloader bytesReceived:(int)bytesReceived expectedLength:(int)expectedLength
{
    if (expectedLength > 0) {
        double percentComplete = ((double)bytesReceived / (double)expectedLength) * 100;
    }
}

- (void)downloaderDidFinishWithData:(Downloader *)downloader data:(NSData *)data
{
    if (downloader == _downloader)
    {
        [self parseAddressData:data];
        if ([downloader isDownloading]) [downloader resetDownload];
    }
}

- (void)downloaderDidReset:(Downloader *)downloader
{
    // Finished, canceled, error, whatever, this method is *always* called when downloader stops.
    // it's the perfect place to update the interface and state.
}


@end
