// Copyright 2016 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.sandbox;

import com.google.devtools.build.lib.vfs.FileSystemUtils;
import com.google.devtools.build.lib.vfs.Path;
import com.google.devtools.build.lib.vfs.PathFragment;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * A description of a subprocess, as well as the necessary file system / sandbox setup.
 *
 * <p>Instances are responsible for making a list of input files available inside the sandbox root,
 * so that a process running inside the directory can access the files. It also handles moving the
 * output files generated by the process out of the directory into a destination directory.
 */
interface SandboxedSpawn {
  /** The path in which to execute the subprocess. */
  Path getSandboxExecRoot();

  /** The command-line of the subprocess. */
  List<String> getArguments();

  /** The environment variables to be set for the subprocess. */
  Map<String, String> getEnvironment();

  /**
   * Creates the sandboxed execution root, making all {@code inputs} available for reading, making
   * sure that the parent directories of all {@code outputs} and that all {@code writableDirs}
   * exist and can be written into.
   *
   * @throws IOException
   */
  void createFileSystem() throws IOException;

  /**
   * Moves all {@code outputs} to {@code execRoot} while keeping the directory structure.
   *
   * @throws IOException
   */
  void copyOutputs(Path execRoot) throws IOException;

  /**
   * Deletes the sandbox directory.
   */
  void delete();

  /**
   * Moves all given outputs from a root to another.
   *
   * <p>This is a support function to help with the implementation of {@link #copyOutputs(Path)}.
   *
   * @param outputs outputs to move as relative paths to a root
   * @param sourceRoot source directory from which to resolve outputs
   * @param targetRoot target directory to which to move the resolved outputs from the source
   * @throws IOException if any of the moves fails
   */
  static void moveOutputs(Collection<PathFragment> outputs, Path sourceRoot, Path targetRoot)
      throws IOException {
    for (PathFragment output : outputs) {
      Path source = sourceRoot.getRelative(output);
      Path target = targetRoot.getRelative(output);
      if (source.isFile() || source.isSymbolicLink()) {
        // Ensure the target directory exists in the target. The directories for the action outputs
        // have already been created, but the spawn outputs may be different from the overall action
        // outputs. This is the case for test actions.
        target.getParentDirectory().createDirectoryAndParents();
        FileSystemUtils.moveFile(source, target);
      } else if (source.isDirectory()) {
        try {
          source.renameTo(target);
        } catch (IOException e) {
          // Failed to move directory directly, thus move it recursively.
          target.createDirectory();
          FileSystemUtils.moveTreesBelow(source, target);
        }
      }
    }
  }
}
