/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.swing.AutoGridBagLayout;
import ccl.swing.SwingUtil;
import ccl.util.Util;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.util.Enumeration;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

/**
 * This dialog provides a basic input facility for a line of text.
 * If a vector of strings is given they can be selected in a drop down list for user convenience.
 */
public class InputCancelDialog extends ExitJDialog 
{
    private JTextField _txtInput = null;
    private OKCancelPanel _pOKCancelPanel= null;
    private String _sValue = "";
    private JComboBox _comboBox = null;

    /**
     * Creates the text field _txtInput with a default text and size.
     */
    private void _createTextField( String sInit_ ) 
    {
        _txtInput = new JTextField(30);
        _txtInput.setBorder( SwingUtil.createCCLBorder() );
        _txtInput.setText( sInit_ );
        SwingUtil.setInitialFocus( this, _getInputComponent() );
    }

    /**
     * Creates and sets up the combo box.
     *
     * @param   sInit_      the selected item.
     * @param   vHistory_   should not be null or an exception will be raised.
     */
    private void _createComboBox( String sInit_, Vector vHistory_, boolean bEditable_ ) 
    {
        _comboBox = new JComboBox();
        _comboBox.setPreferredSize( new Dimension( 30*12, 30 ) );
        _comboBox.setBorder( SwingUtil.createCCLBorder() );
        _comboBox.setEditable( bEditable_ );
        _comboBox.addItem( sInit_ );
        Enumeration eHistory = vHistory_.elements();
        while ( eHistory.hasMoreElements() ) 
        {
            String sNextHistory = (String)eHistory.nextElement();
            if ( !sInit_.equals( sNextHistory ) ) 
            {
                _comboBox.addItem( sNextHistory );
            }
        }
        _comboBox.setSelectedItem( sInit_ );
        SwingUtil.setInitialFocus( this
                                   , _comboBox.getEditor()
                                   .getEditorComponent() );
    }

    /**
     * Returns the input gui component that is actually used, either a JTextField or a JComboBox object.
     */
    private JComponent _getInputComponent() 
    {
        if ( _txtInput != null ) 
        {
            return _txtInput;
        }

        return _comboBox;
    }

    /**
     * Depending on vHistory_ either creates a text field or a combo box for user input.
     * This component is then added to the content pane.
     */
    private void _addInputComponent( String sInit_, Vector vHistory_, boolean bEditable_ ) 
    {
        if ( vHistory_ == null ) 
        {
            _createTextField( sInit_ );
        }
        else 
        {
            _createComboBox( sInit_, vHistory_, bEditable_ );
        }
        getContentPane().add( _getInputComponent() );
    }

    /**
     * Creates an input dialog for a text string.
     *
     * @param   frmParent_   the parent frame.
     * @param   sPrint_      the user request message. A colon and space has to be given by the user,
     *                       it will not be added by the dialog itself.
     * @param   sInit_       the default input value.
     */
    public InputCancelDialog( Frame frmParent_,
                              String sPrint_, 
                              String sInit_ )
    {
        this( frmParent_, sPrint_, sInit_, null );
    }

    /**
     * Creates an input dialog for a text string.
     *
     * @param   frmParent_   the parent frame.
     * @param   sPrint_      the user request message. A colon and space has to be given by the user,
     *                       it will not be added by the dialog itself.
     * @param   sInit_       the default input value.
     * @param   vHistory_    a list of string the user can pick for input.
     *                       If this list is null no drop down list is shown at all.
     *                       The first element in the vector will appear at the top of the list.
     *                       The vector will not be changed. Use updateHistory for that purpose
     *                       after this dialog has been closed.
     */
    public InputCancelDialog( Frame frmParent_,
                              String sPrint_, 
                              String sInit_,
                              Vector vHistory_ )
    {
        this( frmParent_, sPrint_, sInit_, vHistory_, true );
    }

    /**
     * Creates an input dialog for a text string.
     *
     * @param   frmParent_   the parent frame.
     * @param   sPrint_      the user request message. A colon and space has to be given by the user,
     *                       it will not be added by the dialog itself.
     * @param   sInit_       the default input value.
     * @param   vHistory_    a list of string the user can pick for input.
     *                       If this list is null no drop down list is shown at all.
     *                       The first element in the vector will appear at the top of the list.
     *                       The vector will not be changed. Use updateHistory for that purpose
     *                       after this dialog has been closed.
     * @param   bEditable_   false if input is presented via JComboBox and is not editable.
     */
    public InputCancelDialog( Frame frmParent_,
                              String sPrint_, 
                              String sInit_,
                              Vector vHistory_,
                              boolean bEditable_ )
    {
        super( frmParent_, "Input", true );
        Util.panicIf( sPrint_ == null );

        AutoGridBagLayout pAutoGridBagLayout = 
               new AutoGridBagLayout();

        getContentPane().setLayout( pAutoGridBagLayout );

        ((JPanel)getContentPane()).setBorder
               ( BorderFactory.createEmptyBorder( 10, 5, 5, 5 ) );

        getContentPane().add( new JLabel( sPrint_ ) );

        _addInputComponent( sInit_, vHistory_, bEditable_ );
        
        pAutoGridBagLayout.endLine();
        _pOKCancelPanel = new OKCancelPanel( this );
        pAutoGridBagLayout.skip();
        pAutoGridBagLayout.setAnchor( GridBagConstraints.EAST );
        getContentPane().add( _pOKCancelPanel );

        pack();
        SwingUtil.centerComponent(this);
    }

    /**
     * This method indicates if the user did cancel the dialog.
     *
     * @return   false on cancel, otherwise true.
     */
    public boolean isOK() 
    {
        return _pOKCancelPanel.isOK();
    }

    /**
     * This method returns the input from the user.
     */
    public String getValue() 
    {
        if ( !isOK() ) 
        {
            return( (String)null );
        }

        String sRetVal = null;
        if ( _txtInput != null ) 
        {
            sRetVal = _txtInput.getText();
        }
        else 
        {
            sRetVal = _comboBox.getSelectedItem().toString();
        }

        return sRetVal;
    }

    /**
     * This is a convenience method to help keeping the recent history up to date.
     * After 20 lines the rest is dropped off.
     *
     * @param   vHistory_      the old history. This vector will be updated!
     * @param   sNewElement_   the latest new text string that shall be added to the history
     *                         (to the front).
     */
    static public void updateHistory( Vector vHistory_, String sNewElement_ ) 
    {
        if ( !Util.isEmpty( sNewElement_ ) ) 
        {
            vHistory_.removeElement( sNewElement_ );
            vHistory_.insertElementAt( sNewElement_, 0 );
        }
        while( vHistory_.size() > 20 ) 
        {
            vHistory_.removeElementAt( 20 );
        }
    }

    /**
     * Creates an input dialog for a text string.
     *
     * @param   frmParent_   the parent frame.
     * @param   sPrint_      the user request message. A colon and space has to be given by the user,
     *                       it will not be added by the dialog itself.
     * @param   sInit_       the default input value.
     * @param   vHistory_    a list of string the user can pick for input.
     *                       If this list is null no drop down list is shown at all.
     *                       The first element in the vector will appear at the top of the list.
     *                       The vector will be updated when the user did press 'OK'.
     */
    static public String input( Frame frmParent_,
                                String sPrint_, 
                                String sInit_,
                                Vector vHistory_ )
    {
        return input( frmParent_, sPrint_, sInit_, vHistory_, true );
    }

    /**
     * Creates an input dialog for a text string.
     *
     * @param   frmParent_   the parent frame.
     * @param   sPrint_      the user request message. A colon and space has to be given by the user,
     *                       it will not be added by the dialog itself.
     * @param   sInit_       the default input value.
     * @param   vHistory_    a list of string the user can pick for input.
     *                       If this list is null no drop down list is shown at all.
     *                       The first element in the vector will appear at the top of the list.
     *                       The vector will be updated when the user did press 'OK'.
     * @param   bEditable_   false if input is presented via JComboBox and is not editable.
     */
    static public String input( Frame frmParent_,
                                String sPrint_, 
                                String sInit_,
                                Vector vHistory_,
                                boolean bEditable_ )
    {
        String sRetVal = null;
        InputCancelDialog dlgInput = new InputCancelDialog( frmParent_, sPrint_, sInit_, vHistory_, bEditable_ );
        dlgInput.setVisible( true );
        if (dlgInput.isOK())
        {
            sRetVal = dlgInput.getValue();
            updateHistory( vHistory_, sRetVal );
        }
        dlgInput.dispose();
        
        return sRetVal;
    }
}
