/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.watch.registry.impl;

import java.io.File;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import net.rubygrapefruit.platform.NativeException;
import net.rubygrapefruit.platform.file.FileSystemInfo;
import org.gradle.internal.file.FileHierarchySet;
import org.gradle.internal.file.FileMetadata;
import org.gradle.internal.snapshot.FileSystemLocationSnapshot;
import org.gradle.internal.snapshot.FileSystemSnapshotHierarchyVisitor;
import org.gradle.internal.snapshot.SnapshotHierarchy;
import org.gradle.internal.snapshot.SnapshotVisitResult;
import org.gradle.internal.watch.registry.FileWatcherProbeRegistry;
import org.gradle.internal.watch.registry.impl.Combiners;
import org.gradle.internal.watch.vfs.WatchMode;
import org.gradle.internal.watch.vfs.WatchableFileSystemDetector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchableHierarchies {
    private static final Logger LOGGER = LoggerFactory.getLogger(WatchableHierarchies.class);
    public static final String INVALIDATING_HIERARCHY_MESSAGE = "Invalidating hierarchy because watch probe hasn't been triggered";
    private final WatchableFileSystemDetector watchableFileSystemDetector;
    private final FileWatcherProbeRegistry probeRegistry;
    private final Predicate<String> watchFilter;
    private FileHierarchySet watchableFiles = FileHierarchySet.empty();
    private FileHierarchySet unwatchableFiles = FileHierarchySet.empty();
    private final Deque<File> hierarchies = new ArrayDeque<File>();

    public WatchableHierarchies(FileWatcherProbeRegistry probeRegistry, WatchableFileSystemDetector watchableFileSystemDetector, Predicate<String> watchFilter) {
        this.probeRegistry = probeRegistry;
        this.watchableFileSystemDetector = watchableFileSystemDetector;
        this.watchFilter = watchFilter;
    }

    public void registerWatchableHierarchy(File watchableHierarchy, SnapshotHierarchy root) {
        String watchableHierarchyPath = watchableHierarchy.getAbsolutePath();
        if (!this.watchFilter.test(watchableHierarchyPath)) {
            throw new IllegalStateException(String.format("Unable to watch directory '%s' since it is within Gradle's caches", watchableHierarchyPath));
        }
        if (this.unwatchableFiles.contains(watchableHierarchyPath)) {
            LOGGER.info("Not watching {} since the file system is not supported", (Object)watchableHierarchy);
            return;
        }
        if (!this.watchableFiles.contains(watchableHierarchyPath)) {
            this.checkThatNothingExistsInNewWatchableHierarchy(watchableHierarchyPath, root);
            this.hierarchies.addFirst(watchableHierarchy);
            this.watchableFiles = this.watchableFiles.plus(watchableHierarchy);
        } else {
            this.hierarchies.remove(watchableHierarchy);
            this.hierarchies.addFirst(watchableHierarchy);
        }
        LOGGER.info("Now considering {} as hierarchies to watch", this.hierarchies);
    }

    @CheckReturnValue
    public SnapshotHierarchy removeUnwatchableContentOnBuildFinished(SnapshotHierarchy root, WatchMode watchMode, Predicate<File> isWatchedHierarchy, int maximumNumberOfWatchedHierarchies, Invalidator invalidator) {
        SnapshotHierarchy newRoot = this.removeWatchedHierarchiesOverLimit(root, isWatchedHierarchy, maximumNumberOfWatchedHierarchies, invalidator);
        newRoot = this.removeUnwatchedSnapshots(newRoot, invalidator);
        if (!this.shouldWatchUnsupportedFileSystems(watchMode)) {
            newRoot = this.removeUnwatchableFileSystems(newRoot, invalidator);
        }
        return newRoot;
    }

    private SnapshotHierarchy removeUnwatchedSnapshots(SnapshotHierarchy root, Invalidator invalidator) {
        RemoveUnwatchedFiles removeUnwatchedFilesVisitor = new RemoveUnwatchedFiles(root, invalidator);
        root.rootSnapshots().forEach(snapshotRoot -> snapshotRoot.accept((FileSystemSnapshotHierarchyVisitor)removeUnwatchedFilesVisitor));
        return removeUnwatchedFilesVisitor.getRootWithUnwatchedFilesRemoved();
    }

    private SnapshotHierarchy removeWatchedHierarchiesOverLimit(SnapshotHierarchy root, Predicate<File> isWatchedHierarchy, int maximumNumberOfWatchedHierarchies, Invalidator invalidator) {
        this.hierarchies.removeIf(hierarchy -> !isWatchedHierarchy.test((File)hierarchy));
        SnapshotHierarchy result = root;
        int toRemove = this.hierarchies.size() - maximumNumberOfWatchedHierarchies;
        if (toRemove > 0) {
            LOGGER.info("Watching too many directories in the file system (watching {}, limit {}), dropping some state from the virtual file system", (Object)this.hierarchies.size(), (Object)maximumNumberOfWatchedHierarchies);
            for (int i = 0; i < toRemove; ++i) {
                File locationToRemove = this.hierarchies.removeLast();
                result = invalidator.invalidate(locationToRemove.toString(), result);
            }
        }
        this.watchableFiles = this.hierarchies.stream().reduce(FileHierarchySet.empty(), FileHierarchySet::plus, Combiners.nonCombining());
        return result;
    }

    private SnapshotHierarchy removeUnwatchableFileSystems(SnapshotHierarchy root, Invalidator invalidator) {
        SnapshotHierarchy invalidatedRoot = this.invalidateUnsupportedFileSystems(root, invalidator);
        if (invalidatedRoot != root) {
            LOGGER.info("Some of the file system contents retained in the virtual file system are on file systems that Gradle doesn't support watching. The relevant state was discarded to ensure changes to these locations are properly detected. You can override this by explicitly enabling file system watching.");
        }
        return invalidatedRoot;
    }

    public SnapshotHierarchy removeUnwatchableContentOnBuildStart(SnapshotHierarchy root, Invalidator invalidator, WatchMode watchMode) {
        SnapshotHierarchy newRoot = root;
        newRoot = this.removeUnprovenHierarchies(newRoot, invalidator, watchMode);
        return newRoot;
    }

    @Nonnull
    private SnapshotHierarchy removeUnprovenHierarchies(SnapshotHierarchy root, Invalidator invalidator, WatchMode watchMode) {
        return this.probeRegistry.unprovenHierarchies().reduce(root, (currentRoot, unprovenHierarchy) -> {
            if (this.hierarchies.remove(unprovenHierarchy)) {
                watchMode.loggerForWarnings(LOGGER).warn("Invalidating hierarchy because watch probe hasn't been triggered {}", unprovenHierarchy);
                return invalidator.invalidate(unprovenHierarchy.getAbsolutePath(), (SnapshotHierarchy)currentRoot);
            }
            return currentRoot;
        }, Combiners.nonCombining());
    }

    private SnapshotHierarchy invalidateUnsupportedFileSystems(SnapshotHierarchy root, Invalidator invalidator) {
        try {
            return this.watchableFileSystemDetector.detectUnsupportedFileSystems().reduce(root, (updatedRoot, fileSystem) -> invalidator.invalidate(fileSystem.getMountPoint().getAbsolutePath(), (SnapshotHierarchy)updatedRoot), Combiners.nonCombining());
        }
        catch (NativeException e) {
            LOGGER.warn("Unable to list file systems to check whether they can be watched. The whole state of the virtual file system has been discarded. Reason: {}", (Object)e.getMessage());
            return root.empty();
        }
    }

    public Stream<File> stream() {
        return this.hierarchies.stream();
    }

    private void checkThatNothingExistsInNewWatchableHierarchy(String watchableHierarchy, SnapshotHierarchy vfsRoot) {
        vfsRoot.rootSnapshotsUnder(watchableHierarchy).filter(snapshotRoot -> !this.isInWatchableHierarchy(snapshotRoot.getAbsolutePath()) && !this.ignoredForWatching((FileSystemLocationSnapshot)snapshotRoot)).findAny().ifPresent(snapshotRoot -> {
            throw new IllegalStateException(String.format("Found existing snapshot at '%s' for unwatched hierarchy '%s'", snapshotRoot.getAbsolutePath(), watchableHierarchy));
        });
    }

    public boolean ignoredForWatching(FileSystemLocationSnapshot snapshot) {
        return snapshot.getAccessType() == FileMetadata.AccessType.VIA_SYMLINK || !this.watchFilter.test(snapshot.getAbsolutePath());
    }

    public boolean isInWatchableHierarchy(String path) {
        return this.watchableFiles.contains(path);
    }

    public boolean shouldWatch(FileSystemLocationSnapshot snapshot) {
        return !this.ignoredForWatching(snapshot) && this.isInWatchableHierarchy(snapshot.getAbsolutePath());
    }

    public void updateUnsupportedFileSystems(WatchMode watchMode) {
        this.unwatchableFiles = this.shouldWatchUnsupportedFileSystems(watchMode) ? FileHierarchySet.empty() : this.detectUnsupportedHierarchies();
    }

    private boolean shouldWatchUnsupportedFileSystems(WatchMode watchMode) {
        return watchMode != WatchMode.DEFAULT;
    }

    private FileHierarchySet detectUnsupportedHierarchies() {
        try {
            return this.watchableFileSystemDetector.detectUnsupportedFileSystems().map(FileSystemInfo::getMountPoint).reduce(FileHierarchySet.empty(), FileHierarchySet::plus, Combiners.nonCombining());
        }
        catch (NativeException e) {
            LOGGER.warn("Unable to list file systems to check whether they can be watched. Assuming all file systems can be watched. Reason: {}", (Object)e.getMessage());
            return FileHierarchySet.empty();
        }
    }

    public static interface Invalidator {
        public SnapshotHierarchy invalidate(String var1, SnapshotHierarchy var2);
    }

    private class RemoveUnwatchedFiles
    implements FileSystemSnapshotHierarchyVisitor {
        private SnapshotHierarchy root;
        private final Invalidator invalidator;

        public RemoveUnwatchedFiles(SnapshotHierarchy root, Invalidator invalidator) {
            this.root = root;
            this.invalidator = invalidator;
        }

        public SnapshotVisitResult visitEntry(FileSystemLocationSnapshot snapshot) {
            if (this.shouldBeRemoved(snapshot)) {
                this.invalidateUnwatchedFile(snapshot);
                return SnapshotVisitResult.SKIP_SUBTREE;
            }
            return SnapshotVisitResult.CONTINUE;
        }

        private boolean shouldBeRemoved(FileSystemLocationSnapshot snapshot) {
            return snapshot.getAccessType() == FileMetadata.AccessType.VIA_SYMLINK || !WatchableHierarchies.this.isInWatchableHierarchy(snapshot.getAbsolutePath()) && WatchableHierarchies.this.watchFilter.test(snapshot.getAbsolutePath());
        }

        private void invalidateUnwatchedFile(FileSystemLocationSnapshot snapshot) {
            this.root = this.invalidator.invalidate(snapshot.getAbsolutePath(), this.root);
        }

        public SnapshotHierarchy getRootWithUnwatchedFilesRemoved() {
            return this.root;
        }
    }
}

