/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.hyracks.api.exceptions;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.Serializable;
import java.util.Objects;

public final class SourceLocation implements Serializable {

    private static final long serialVersionUID = 1L;

    private final int line;

    private final int column;

    public SourceLocation(int line, int column) {
        this.line = line;
        this.column = column;
    }

    public int getLine() {
        return line;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SourceLocation that = (SourceLocation) o;
        return line == that.line && column == that.column;
    }

    @Override
    public int hashCode() {
        return Objects.hash(line, column);
    }

    public int getColumn() {
        return column;
    }

    public static void writeFields(SourceLocation sourceLocation, DataOutput output) throws IOException {
        if (sourceLocation == null) {
            output.writeInt(-1);
        } else {
            output.writeInt(sourceLocation.getLine());
            output.writeInt(sourceLocation.getColumn());
        }
    }

    public static SourceLocation create(DataInput dataInput) throws IOException {
        int row = dataInput.readInt();
        if (row == -1) {
            return null;
        }
        return new SourceLocation(row, dataInput.readInt());
    }
}
