/*
 * Copyright (c) Citrix Systems, Inc.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *   1) Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 * 
 *   2) Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */


using System;
using System.Collections;
using System.Collections.Generic;
using System.Management.Automation;
using System.Text;

using XenAPI;

namespace Citrix.XenServer.Commands
{
    [Cmdlet(VerbsCommon.Get, "XenVIFProperty", SupportsShouldProcess=false)]
    public class GetXenVIFProperty : XenServerCmdlet
    {
        #region Cmdlet Parameters

        [Parameter(ParameterSetName = "XenObject", Mandatory = true, ValueFromPipeline = true, Position = 0)]
        public XenAPI.VIF VIF { get; set; }
        
        [Parameter(ParameterSetName = "Ref", Mandatory = true, ValueFromPipelineByPropertyName = true, Position = 0)]
        [Alias("opaque_ref")]
        public XenRef<XenAPI.VIF> Ref { get; set; }


        [Parameter(Mandatory = true)]
        public XenVIFProperty XenProperty { get; set; }
        
        #endregion

        #region Cmdlet Methods
        
        protected override void ProcessRecord()
        {
            GetSession();
            
            string vif = ParseVIF();
            
            switch (XenProperty)
            {
                case XenVIFProperty.Uuid:
                    ProcessRecordUuid(vif);
                    break;
                case XenVIFProperty.AllowedOperations:
                    ProcessRecordAllowedOperations(vif);
                    break;
                case XenVIFProperty.CurrentOperations:
                    ProcessRecordCurrentOperations(vif);
                    break;
                case XenVIFProperty.Device:
                    ProcessRecordDevice(vif);
                    break;
                case XenVIFProperty.Network:
                    ProcessRecordNetwork(vif);
                    break;
                case XenVIFProperty.VM:
                    ProcessRecordVM(vif);
                    break;
                case XenVIFProperty.MAC:
                    ProcessRecordMAC(vif);
                    break;
                case XenVIFProperty.MTU:
                    ProcessRecordMTU(vif);
                    break;
                case XenVIFProperty.OtherConfig:
                    ProcessRecordOtherConfig(vif);
                    break;
                case XenVIFProperty.CurrentlyAttached:
                    ProcessRecordCurrentlyAttached(vif);
                    break;
                case XenVIFProperty.StatusCode:
                    ProcessRecordStatusCode(vif);
                    break;
                case XenVIFProperty.StatusDetail:
                    ProcessRecordStatusDetail(vif);
                    break;
                case XenVIFProperty.RuntimeProperties:
                    ProcessRecordRuntimeProperties(vif);
                    break;
                case XenVIFProperty.QosAlgorithmType:
                    ProcessRecordQosAlgorithmType(vif);
                    break;
                case XenVIFProperty.QosAlgorithmParams:
                    ProcessRecordQosAlgorithmParams(vif);
                    break;
                case XenVIFProperty.QosSupportedAlgorithms:
                    ProcessRecordQosSupportedAlgorithms(vif);
                    break;
                case XenVIFProperty.Metrics:
                    ProcessRecordMetrics(vif);
                    break;
                case XenVIFProperty.MACAutogenerated:
                    ProcessRecordMACAutogenerated(vif);
                    break;
                case XenVIFProperty.LockingMode:
                    ProcessRecordLockingMode(vif);
                    break;
                case XenVIFProperty.Ipv4Allowed:
                    ProcessRecordIpv4Allowed(vif);
                    break;
                case XenVIFProperty.Ipv6Allowed:
                    ProcessRecordIpv6Allowed(vif);
                    break;
            }
            
            UpdateSessions();
        }
        
        #endregion
    
        #region Private Methods

        private string ParseVIF()
        {
            string vif = null;

            if (VIF != null)
                vif = (new XenRef<XenAPI.VIF>(VIF)).opaque_ref;
            else if (Ref != null)
                vif = Ref.opaque_ref;
            else
            {
                ThrowTerminatingError(new ErrorRecord(
                    new ArgumentException("At least one of the parameters 'VIF', 'Ref', 'Uuid' must be set"),
                    string.Empty,
                    ErrorCategory.InvalidArgument,
                    VIF));
            }

            return vif;
        }

        private void ProcessRecordUuid(string vif)
        {
            RunApiCall(()=>
            {
                    string obj = XenAPI.VIF.get_uuid(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordAllowedOperations(string vif)
        {
            RunApiCall(()=>
            {
                    List<vif_operations> obj = XenAPI.VIF.get_allowed_operations(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordCurrentOperations(string vif)
        {
            RunApiCall(()=>
            {
                    var dict = XenAPI.VIF.get_current_operations(session, vif);

                        Hashtable ht = CommonCmdletFunctions.ConvertDictionaryToHashtable(dict);
                        WriteObject(ht, true);
            });
        }

        private void ProcessRecordDevice(string vif)
        {
            RunApiCall(()=>
            {
                    string obj = XenAPI.VIF.get_device(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordNetwork(string vif)
        {
            RunApiCall(()=>
            {
                    string objRef = XenAPI.VIF.get_network(session, vif);

                        XenAPI.Network obj = null;

                        if (objRef != "OpaqueRef:NULL")
                        {
                            obj = XenAPI.Network.get_record(session, objRef);
                            obj.opaque_ref = objRef;
                        }

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordVM(string vif)
        {
            RunApiCall(()=>
            {
                    string objRef = XenAPI.VIF.get_VM(session, vif);

                        XenAPI.VM obj = null;

                        if (objRef != "OpaqueRef:NULL")
                        {
                            obj = XenAPI.VM.get_record(session, objRef);
                            obj.opaque_ref = objRef;
                        }

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordMAC(string vif)
        {
            RunApiCall(()=>
            {
                    string obj = XenAPI.VIF.get_MAC(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordMTU(string vif)
        {
            RunApiCall(()=>
            {
                    long obj = XenAPI.VIF.get_MTU(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordOtherConfig(string vif)
        {
            RunApiCall(()=>
            {
                    var dict = XenAPI.VIF.get_other_config(session, vif);

                        Hashtable ht = CommonCmdletFunctions.ConvertDictionaryToHashtable(dict);
                        WriteObject(ht, true);
            });
        }

        private void ProcessRecordCurrentlyAttached(string vif)
        {
            RunApiCall(()=>
            {
                    bool obj = XenAPI.VIF.get_currently_attached(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordStatusCode(string vif)
        {
            RunApiCall(()=>
            {
                    long obj = XenAPI.VIF.get_status_code(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordStatusDetail(string vif)
        {
            RunApiCall(()=>
            {
                    string obj = XenAPI.VIF.get_status_detail(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordRuntimeProperties(string vif)
        {
            RunApiCall(()=>
            {
                    var dict = XenAPI.VIF.get_runtime_properties(session, vif);

                        Hashtable ht = CommonCmdletFunctions.ConvertDictionaryToHashtable(dict);
                        WriteObject(ht, true);
            });
        }

        private void ProcessRecordQosAlgorithmType(string vif)
        {
            RunApiCall(()=>
            {
                    string obj = XenAPI.VIF.get_qos_algorithm_type(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordQosAlgorithmParams(string vif)
        {
            RunApiCall(()=>
            {
                    var dict = XenAPI.VIF.get_qos_algorithm_params(session, vif);

                        Hashtable ht = CommonCmdletFunctions.ConvertDictionaryToHashtable(dict);
                        WriteObject(ht, true);
            });
        }

        private void ProcessRecordQosSupportedAlgorithms(string vif)
        {
            RunApiCall(()=>
            {
                    string[] obj = XenAPI.VIF.get_qos_supported_algorithms(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordMetrics(string vif)
        {
            RunApiCall(()=>
            {
                    string objRef = XenAPI.VIF.get_metrics(session, vif);

                        XenAPI.VIF_metrics obj = null;

                        if (objRef != "OpaqueRef:NULL")
                        {
                            obj = XenAPI.VIF_metrics.get_record(session, objRef);
                            obj.opaque_ref = objRef;
                        }

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordMACAutogenerated(string vif)
        {
            RunApiCall(()=>
            {
                    bool obj = XenAPI.VIF.get_MAC_autogenerated(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordLockingMode(string vif)
        {
            RunApiCall(()=>
            {
                    vif_locking_mode obj = XenAPI.VIF.get_locking_mode(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordIpv4Allowed(string vif)
        {
            RunApiCall(()=>
            {
                    string[] obj = XenAPI.VIF.get_ipv4_allowed(session, vif);

                        WriteObject(obj, true);
            });
        }

        private void ProcessRecordIpv6Allowed(string vif)
        {
            RunApiCall(()=>
            {
                    string[] obj = XenAPI.VIF.get_ipv6_allowed(session, vif);

                        WriteObject(obj, true);
            });
        }

        #endregion
    }
    
    public enum XenVIFProperty
    {
        Uuid,
        AllowedOperations,
        CurrentOperations,
        Device,
        Network,
        VM,
        MAC,
        MTU,
        OtherConfig,
        CurrentlyAttached,
        StatusCode,
        StatusDetail,
        RuntimeProperties,
        QosAlgorithmType,
        QosAlgorithmParams,
        QosSupportedAlgorithms,
        Metrics,
        MACAutogenerated,
        LockingMode,
        Ipv4Allowed,
        Ipv6Allowed
    }

}
