/*
Copyright 2022 The Tekton Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package checksum

import (
	"encoding/hex"
	"testing"

	"github.com/google/go-cmp/cmp"

	"github.com/tektoncd/pipeline/test/diff"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

func TestPrepareObjectMeta(t *testing.T) {
	unsigned := metav1.ObjectMeta{
		Name:        "test-task",
		Annotations: map[string]string{"foo": "bar"},
	}
	namespace := ""
	signed := unsigned.DeepCopy()
	sig := "tY805zV53PtwDarK3VD6dQPx5MbIgctNcg/oSle+MG0="
	signed.Annotations = map[string]string{SignatureAnnotation: sig}

	signedWithLabels := signed.DeepCopy()
	signedWithLabels.Labels = map[string]string{"label": "foo"}

	signedWithExtraAnnotations := signed.DeepCopy()
	signedWithExtraAnnotations.Annotations["kubectl-client-side-apply"] = "client"
	signedWithExtraAnnotations.Annotations["kubectl.kubernetes.io/last-applied-configuration"] = "config"

	tcs := []struct {
		name       string
		objectmeta *metav1.ObjectMeta
		expected   metav1.ObjectMeta
	}{{
		name:       "Prepare signed objectmeta without labels",
		objectmeta: signed,
		expected: metav1.ObjectMeta{
			Name:        "test-task",
			Namespace:   namespace,
			Annotations: map[string]string{},
		},
	}, {
		name:       "Prepare signed objectmeta with labels",
		objectmeta: signedWithLabels,
		expected: metav1.ObjectMeta{
			Name:        "test-task",
			Namespace:   namespace,
			Labels:      map[string]string{"label": "foo"},
			Annotations: map[string]string{},
		},
	}, {
		name:       "Prepare signed objectmeta with extra annotations",
		objectmeta: signedWithExtraAnnotations,
		expected: metav1.ObjectMeta{
			Name:        "test-task",
			Namespace:   namespace,
			Annotations: map[string]string{},
		},
	}, {
		name:       "resource without signature shouldn't fail",
		objectmeta: &unsigned,
		expected: metav1.ObjectMeta{
			Name:        "test-task",
			Namespace:   namespace,
			Annotations: map[string]string{"foo": "bar"},
		},
	}}

	for _, tc := range tcs {
		t.Run(tc.name, func(t *testing.T) {
			task := PrepareObjectMeta(tc.objectmeta)
			if d := cmp.Diff(tc.expected, task); d != "" {
				t.Error(diff.PrintWantGot(d))
			}
		})
	}
}

func TestComputeSha256Checksum(t *testing.T) {
	sha, err := ComputeSha256Checksum("hello")
	if err != nil {
		t.Fatalf("Could not marshal hello %v", err)
	}
	if d := cmp.Diff("5aa762ae383fbb727af3c7a36d4940a5b8c40a989452d2304fc958ff3f354e7a", hex.EncodeToString(sha)); d != "" {
		t.Error(diff.PrintWantGot(d))
	}
}
